
from skimage.metrics import structural_similarity as ssim
import argparse
import imutils
import numpy as np
import cv2
import json
from os.path import exists
import logging
from rttkpkg import logger as rttkloger
from rttkpkg import session as sess
log = logging.getLogger(rttkloger.LoggerName('ScreenDiff'))


def StepDiffResultName(stepid):
    return('Analysis/ScreenDiff/'+stepid)

def SaveDiffToSession(rs, stepid, stepdr):
    if rs.FileExistsInSession(StepDiffResultName(stepid)):
        log.critical('Diff already exists remove it first')
    else:
        rs.AddFileToSession(StepDiffResultName(stepid), stepdr)

def ExportDiffAreas(cnts):
    dl = []
    for c in cnts:
	# compute the bounding box of the contour and then draw the
	# bounding box on both input images to represent where the two
	# images differ
        (x, y, w, h) = cv2.boundingRect(c)
        br = {'x': x, 'y': y, 'w':w, "h":h}
        dl.append(br)
        log.debug("area bounds: {}".format(json.dumps(br)))
    return(dl)

def AnalyzeScreenPair(rs, imgBefore, imgAfter, sn):
    lr ={"ScreenName" : sn}
    zimA = rs.GetBitmap(imgAfter)
    zimB = rs.GetBitmap(imgBefore)
    imageA = cv2.imdecode(np.frombuffer(zimA, np.uint8), 1)
    imageB = cv2.imdecode(np.frombuffer(zimB, np.uint8), 1)
    # convert the images to grayscale
    grayA = cv2.cvtColor(imageA, cv2.COLOR_BGR2GRAY)
    grayB = cv2.cvtColor(imageB, cv2.COLOR_BGR2GRAY)
    # compute the Structural Similarity Index (SSIM) between the two
    # images, ensuring that the difference image is returned
    (score, diff) = ssim(grayA, grayB, full=True)
    diff = (diff * 255).astype("uint8")
    lr["Score"] = score
    log.debug("Screen {}".format(sn) +" score SSIM: {}".format(score))
    if score < 1 :
        thresh = cv2.threshold(diff, 0, 255, cv2.THRESH_BINARY_INV | cv2.THRESH_OTSU)[1]
        cnts = cv2.findContours(thresh.copy(), cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
        cnts = imutils.grab_contours(cnts)
        lr["Diffs"]= ExportDiffAreas(cnts)
    return(lr)

def AnalyzeScreenDiffs(rs, si_diffs):
    dpr =[]
    for dp in si_diffs:
        imgBefore = dp["BeforeGuidStr"]
        imgAfter = dp["AfterGuidStr"]
        sn = dp["ScreenName"]
        dpr.append(AnalyzeScreenPair(rs, imgBefore, imgAfter, sn))
    return(dpr)

def AnalyzeStepInspector(rs, si_guid):
    log.debug('Processing step id {}'.format(si_guid))
    if rs.FileExistsInSession(StepDiffResultName(si_guid)):
        log.error('Analysis Exists, skipping {}'.format(si_guid))
    else:
        log.debug('Analyzing step {}'.format(si_guid))
        try:
            sientry = rs.StepDetail(si_guid)
            sijs = json.loads(sientry)
            si_dpr = AnalyzeScreenDiffs(rs, sijs["ScreenDiffs"])
            SaveDiffToSession(rs, si_guid, si_dpr)
            log.debug('Setp Analysis completed{}'.format(si_guid))
        except  Exception as e:
            log.error('Setp Analysis Failed {}'.format(e))

def ProcessSteps(rs):
    log.debug('Processing Steps looking for inspection')
    for obj in rs.StepInfos:
        if obj["ClassTypeName"] == "TSESSessionStepInspector":
            ssi_id = obj["ID"]
            log.debug('Located step for evaluation {}'.format(ssi_id))
            AnalyzeStepInspector(rs, ssi_id)

def ScreenCompare(sessionfile):
    try:
        rs = sess.RttkSession(sessionfile)
        log.info('Comparing Screens for Session File = {}'.format(rs.SessionFile) )
        if rs.IsValid():
            rs.MountSession()
            ProcessSteps(rs)
            del rs
            return True
        else:
            log.critical('Session valid status={}'.format(rs.IsValid()))
            return False
    except  Exception as e:
        log.debug('Error in loading session : {}'.format(e))
        return False

def validateinstall():
    try:
        import argparse
        log.debug('ArgParse imported')
        import zipfile
        log.debug('ZipFile imported')
        import zlib
        log.debug('zlib imported')
        import json
        log.debug('Json Imported')
        import imutils
        log.debug('imutils imported')
        import numpy as np
        log.debug('numpy imported')
        import cv2
        log.debug('cv2 imported')
        import skimage.metrics
        log.debug('Structural similarity imported')
        import sys
        log.debug('Sys path ={}'.format(sys.path))
        return True
    except ModuleNotFoundError as err:
        # Error handling
        log.critical("Module not found Exception :{}".format(err))
        return False
    except  Exception as e:
        log.critical('Error in validate install :{}'.format(e))
        return False


if __name__ == "__main__":
    rttkloger.RunAsScript()
    ap = argparse.ArgumentParser()
    ap.add_argument("-s", "--session", required=True,	help="Session File Full Path")
    args = vars(ap.parse_args())
    ScreenCompare(args["session"])
