from datetime import datetime, timedelta
from genericpath import exists
import stat
import paramiko
import argparse
import socket
import shutil
import os
from rttkpkg import ospaths as rttkospaths
import logging
from rttkpkg import logger as rttkloger
log = logging.getLogger(rttkloger.LoggerName('rttkSSH'))

class RTTK_SSH_CMD:
   def __init__(self):
      self.cmd_str = None
      self.cmd_error = None
      self.cmd_output = None
      self.cmd_result = False

   def SetCmd(self, CmdStr):
      self.cmd_str = CmdStr

   def SetResult(self, CmdError, CmdOutput, CmdResult):
      self.cmd_error = CmdError
      self.cmd_output = CmdOutput
      self.cmd_result = CmdResult



class RTTK_SSH_Util:
   "SSH client for RTTK file collection"

   def __init__(self):
      self.ssh_output = None
      self.ssh_error = None
      self.ssh_input = None
      self.userdir = None
      self.timeout = 5
      self.port = 22
      self.client = paramiko.SSHClient()
      #Parsing an instance of the AutoAddPolicy to set_missing_host_key_policy() changes it to allow any host.
      self.client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
      log.debug('SSH Util class Created')

   def __del__(self):
      self.disconnect()
      log.debug('SSH Util class destroyed')

   def ConfigureDirectories(self, LocalDir):
        if exists(LocalDir):
            self.downloadlocalfilepath = LocalDir
        else:
            raise ValueError('Local directory does not exist : {}'.format(LocalDir))

   def ConfigureHost(self, Hostname, Username, Password):
      self.host= Hostname
      self.username = Username
      self.password = Password
      self.__authpass = True

   def ConfigureHostAuth(self, Hostname, Username):
      self.host= Hostname
      self.username = Username
      self.__authpass = False

   def connect(self):
      # "Login to the remote server"
      try:
      #Paramiko.SSHClient can be used to make connections to the remote server and transfer files
         log.debug("Establishing ssh connection...")
            #Connect to the server
         if self.__authpass == False:
            self.client.connect(hostname=self.host, port=self.port, username=self.username, timeout=self.timeout)
            log.debug("Connected to the server {}".format(self.host))
         else:
            self.client.connect(hostname=self.host, port=self.port,username=self.username,password=self.password,timeout=self.timeout, allow_agent=False, look_for_keys=False)
            log.debug("Connected to the server {}".format(self.host))
      except paramiko.AuthenticationException as ae:
         log.critical("Authentication failed, please verify your credentials: {}".format(ae))
         result_flag = False
      except paramiko.SSHException as sshException:
         log.error("Could not establish SSH connection: {}".format(sshException))
         result_flag = False
      except socket.timeout as e:
         log.error("Connection timed out {}".format(e))
         result_flag = False
      except Exception as e:
         log.error('Exception in connecting to the server :{}'.format(e))
         result_flag = False
         self.client.close()
      else:
         result_flag = True
      return result_flag

   def disconnect(self):
      try:
         self.client.close()
         log.debug('SSH Util client disconnected')
      except Exception as e:
         log.info("error in disconnect :{}".format(e))

   def connected(self):
      try:
         tp = self.client.get_transport()
         if tp is None:
            result_flag = False
            log.debug('Connection does not have transport')
         else:
            result_flag =  tp.is_authenticated()
            log.debug('connection is {}'.format(result_flag))
      except Exception as e:
         log.info("Exception in connection Check :{}".format(e))
         result_flag = False
      return result_flag

   def conncheck(self):
      result_flag = self.connected()
      if result_flag == False:
         log.error('not connected attempting reconnect')
         self.disconnect()
         self.connect()
         result_flag = self.connected()
      return result_flag

   def execute_command(self,command, cmdresult):
      self.ssh_output = None
      self.ssh_error = None
      result_flag = True
      try:
         if self.conncheck():
            log.debug("Executing command --> {}".format(command))
            stdin, stdout, stderr = self.client.exec_command(command,timeout=10)
            self.ssh_error = stderr.read().decode().strip()
            if len(self.ssh_error) > 0 :
               log.error("Problem occurred while running command: The error is {}".format(self.ssh_error))
               cmdresult = self.ssh_error
               result_flag = False
            else:
               log.debug("Command execution completed successfully :{}".format(command))
               self.ssh_output = stdout.read().decode().strip()
               cmdresult = self.ssh_output
               log.debug("ouput ={}".format(str(self.ssh_output)))
               # self.client.close()
         else:
            log.error("Could not establish SSH connection")
            result_flag = False
      except socket.timeout as e:
            log.error("socket error :{}".format(e))
            self.client.close()
            result_flag = False
      except paramiko.SSHException:
            log.error("Failed to execute the command : {}".format(command))
            self.client.close()
            result_flag = False
      return result_flag

   def exec_cmd(self, rttksshcmd):
      rttksshcmd.cmd_output = None
      rttksshcmd.cmd_error = None
      rttksshcmd.cmd_result = False
      try:
         if self.conncheck():
            stdin, stdout, stderr = self.client.exec_command(rttksshcmd.cmd_str,timeout=10)
            rttksshcmd.cmd_error = stderr.read().decode().strip()
            if len(rttksshcmd.cmd_error) > 0 :
               log.error("Problem occurred while running command:{}".format(rttksshcmd.cmd_str) + " The error is: ".format( self.ssh_error))
               rttksshcmd.cmd_result = False
            else:
               log.debug("Command execution completed successfully:{}".format(rttksshcmd.cmd_str) )
               rttksshcmd.cmd_output = stdout.read().decode().strip()
               log.debug("Command output :{}".format(rttksshcmd.cmd_output))
               rttksshcmd.cmd_result = True
         else:
            rttksshcmd.cmd_error = "Could not establish SSH connection"
            rttksshcmd.cmd_result = False
      except socket.timeout as e:
            log.error("socket error :{}".format(e))
            rttksshcmd.cmd_error = "socket error :{}".format(e)
            self.client.close()
            rttksshcmd.cmd_result = False
      except paramiko.SSHException as se:
            log.error('SSH exception {}'.format(se))
            rttksshcmd.cmd_error ="Failed to execute the command!{}".format(rttksshcmd.cmd_str)
            self.client.close()
            rttksshcmd.cmd_result = False
      return rttksshcmd.cmd_result

   def download_file(self,downloadremotefilepath,downloadlocalfilepath):
        try:
            if self.connect():
                ftp_client= self.client.open_sftp()
                ftp_client.get(downloadremotefilepath,downloadlocalfilepath)
                ftp_client.close()
                self.client.close()
                result_flag = True
            else:
                log.error("Could not establish SSH connection")
                result_flag = False
        except Exception as e:
            log.error('Unable to download {}'.format(downloadremotefilepath))
            log.error('Exception was: {}'.format(e))
            result_flag = False
            ftp_client.close()
            self.client.close()
        return result_flag

   def UserCacheDir(self):
      cmd = RTTK_SSH_CMD()
      cmd.SetCmd("pwd")
      if self.exec_cmd(cmd):
         ud = cmd.cmd_output +'/Library/Caches/RTTK'
         log.debug("Cache Dir set to :{}".format(ud))
         return ud
      else:
         raise ValueError("could not get the directory")

   def MoveLocal(self, ftpClient, srvPath, flName):
      lclfn = self.downloadlocalfilepath + '/'+flName
      lclTemp = lclfn +'.transit'
      if exists(lclTemp):
         log.debug('local tranist file exists, attempting to delete')
         try:
            shutil.rmtree(lclTemp)
            log.debug('File removed {}'.format(lclTemp))
         except Exception as e:
            log.error('remove failed for file {}'.format(lclTemp))
      log.debug('Moving {}'.format(srvPath+'/'+flName) + ' to {}'.format(lclfn))
      try:
         ftpClient.chdir(srvPath)
         ftpClient.get(flName, lclTemp)
         shutil.move(lclTemp, lclfn)
         ftpClient.remove(flName)
      except Exception as e:
         log.critical("could not move file{}".format(srvPath+'/'+flName))
         log.info('move local error is {}'.format(e))

   def UserRttkSessList(self):
      if self.connected:
         log.debug('starting list files')
         ftp_client= self.client.open_sftp()
         ucd = self.UserCacheDir()
         log.info('Cache dir = {}'.format(ucd))
         exts = [".rttksess",".rttkinsp"]
         dirfiles = ftp_client.listdir_attr(ucd)
         for fl in dirfiles:
            fn, fe = os.path.splitext(fl.filename)
            if fe in exts:
               last_modified = datetime.fromtimestamp(fl.st_mtime)
               if (datetime.now()-last_modified)>=timedelta(seconds=45):
                  log.info('Moving file ' + fl.filename + 'nm' +fn)
                  self.MoveLocal(ftp_client, ucd, fl.filename )
      else:
         log.error('Not connected - aborting list files')

def ValidateArgs(args):
   if os.path.exists(args["localdir"]) == False:
      log.critical('Local directory does not exist:{}'.format(args["localdir"]))
      raise ValueError('Local directory not found')

def CollectSessionsOSX(localdir, hostname, username):
    try:
        sc = RTTK_SSH_Util()
        sc.ConfigureDirectories(localdir)
        sc.ConfigureHostAuth(hostname, username)
        if sc.connect():
            log.debug('connected to {}'.format(hostname))
            sc.UserRttkSessList()
            return True
        else:
            return False
        del sc
    except Exception as e:
        log.critical('failed in colleting sessions {}'.format(e))
        return False

def CollectSessions(args):
   try:
      rttk_ssh = RTTK_SSH_Util()
      rttk_ssh.ConfigureDirectories(args["localdir"])
      rttk_ssh.ConfigureHost(args["servername"], args["username"], args["password"])
      rttk_ssh.connect()
      log.debug('Listing files to retrieve')
      rttk_ssh.UserRttkSessList()
      del rttk_ssh
   except Exception as e:
      log.critical('failed in colleting sessions {}'.format(e))

def ValidateAgentAuth(hostname, username):
    try:
        client = paramiko.SSHClient()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        client.connect(hostname, 22, username, timeout=5)
        log.info('connected')
        client.close()
    except paramiko.AuthenticationException as ae:
        log.critical("Authentication failed, please verify your credentials: {}".format(ae))
        result_flag = False
    except paramiko.SSHException as sshException:
        log.error("Could not establish SSH connection: {}".format(sshException))
        result_flag = False
    except socket.timeout as e:
        log.error("Connection timed out {}".format(e))
        result_flag = False
    except Exception as e:
        log.error('Exception in connecting to the server :{}'.format(e))
        result_flag = False

if __name__ == "__main__":
   rttkloger.RunAsScript()
   ap = argparse.ArgumentParser()
   ap.add_argument("-s", "--servername", required=True, help="OSX hostname")
   ap.add_argument("-u", "--username", required=True,	help="Username for OSX")
   ap.add_argument("-p", "--password", required=True,	help="Password for OSX")
   ap.add_argument("-ld", "--localdir", required=True, help="Local Directory to move files to")
   try:
      args = vars(ap.parse_args())
      ValidateArgs(args)
      log.debug('Args validated, starting collection')
      CollectSessions(args)
   except Exception as e:
      log.debug('Error running as script: {}'.format(e))

